package com.example.demo;

import com.example.demo.domain.Article;
import com.example.demo.domain.Basket;
import com.example.demo.domain.CardItem;
import com.example.demo.domain.dto.ArticleRequest;
import com.example.demo.repository.IArticleRepository;
import com.example.demo.repository.IBasketRepository;
import com.example.demo.repository.ICardRepository;
import com.example.demo.service.contract.IShopService;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.http.MediaType;
import org.springframework.test.context.DynamicPropertyRegistry;
import org.springframework.test.context.DynamicPropertySource;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.MvcResult;
import org.springframework.test.web.servlet.request.MockMvcRequestBuilders;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.testcontainers.containers.MySQLContainer;
import org.testcontainers.junit.jupiter.Container;
import org.testcontainers.junit.jupiter.Testcontainers;

import java.util.ArrayList;
import java.util.List;

import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static org.springframework.security.test.web.servlet.request.SecurityMockMvcRequestPostProcessors.user;

@SpringBootTest
@Testcontainers
@AutoConfigureMockMvc
public class DemoApplicationTests {
    @Autowired
    private MockMvc mockMvc;
    @Autowired
    private ObjectMapper mapper;

    @Autowired
    private IShopService shopService;

    @Autowired
    private IArticleRepository articleRepository;

    @Autowired
    private IBasketRepository basketRepository;

    @Autowired
    private ICardRepository cardRepository;

    @Container
    static MySQLContainer mySQLContainer =
            new MySQLContainer("mysql:5:7:37");

    @DynamicPropertySource
    static void registerMySQLProperties(DynamicPropertyRegistry registry) {
        registry.add("spring.datasource.url", mySQLContainer::getJdbcUrl);
        registry.add("spring.datasource.username", mySQLContainer::getUsername);
        registry.add("spring.datasource.password", mySQLContainer::getPassword);
    }

    private Article article1;
    private Article article2;
    private CardItem cardItem;
    private Basket basket;

    @BeforeEach
    public void setup(){
        basketRepository.deleteAll();
        cardRepository.deleteAll();;
        articleRepository.deleteAll();

        article1 = Article.builder()
                .name("2x2mm")
                .ean("ASDFASFWEEER")
                .price(0.8)
                .build();
        article2 = Article.builder()
                .name("3x3mm")
                .ean("DKGJORJR")
                .price(0.8)
                .build();
        article1 = articleRepository.save(article1);
        article2 = articleRepository.save(article2);

        cardItem = CardItem.builder()
                .amount(5)
                .article(article1)
                .build();

        List<CardItem> cardItems = new ArrayList<CardItem>();
        cardItems.add(cardItem);
        basket = Basket.builder().payed(false).items(cardItems).build();
        basket = basketRepository.save(basket);

//        Assertions.assertEquals(2, articleRepository.findAll().size());
//        Assertions.assertEquals(1, cardRepository.findAll().size());
//        Assertions.assertEquals(1, basketRepository.findAll().size());
    }

    @Test
    public void addItemToBasket() throws Exception {
        ArticleRequest request = ArticleRequest.builder()
                .name("Screw 4x4mm")
                .price(0.7)
                .ean("ASSDFASDFQE")
                .build();

        String requestString = mapper.writeValueAsString(request);

        mockMvc.perform(MockMvcRequestBuilders.post("/article")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(requestString)
                        .with(user("test").roles("USER")))
                .andExpect(status().isOk());

        Assertions.assertEquals(3, articleRepository.findAll().size());
    }

    @Test
    public void removeArticleFromBasket() throws Exception {
        mockMvc.perform(MockMvcRequestBuilders.delete("/shopping/basket/" + basket.getId() + "/" + basket.getItems().get(0).getId())
                .with(user("test").roles("USER")))
                .andExpect(status().isOk());

        Assertions.assertEquals(0, basketRepository.findById(basket.getId()).get().getItems().size());
    }

    @Test
    public void getBasket() throws Exception {
        MvcResult a = mockMvc.perform(MockMvcRequestBuilders.get("/shopping/basket/" + basket.getId())
                .with(user("test").roles("USER")))
                .andExpect(status().isOk()).andReturn();
    }
}
