package com.example.demo.service;

import com.example.demo.domain.Article;
import com.example.demo.domain.Basket;
import com.example.demo.domain.CardItem;
import com.example.demo.domain.dto.BasketAddItemRequest;
import com.example.demo.repository.IArticleRepository;
import com.example.demo.repository.IBasketRepository;
import com.example.demo.repository.ICardRepository;
import org.junit.Assert;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.*;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.web.servlet.MockMvc;
import java.util.Optional;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
public class ShopServiceTests {
    final String name = "4x4mm";
    final Double price = 0.8;
    final String ean = "ASDFK355SSER";

    @Autowired
    private MockMvc mockMvc;

    @InjectMocks
    private ShopService shopService;

    @Mock
    private IArticleRepository articleRepository;

    @Mock
    private ICardRepository cardRepository;

    @Mock
    private IBasketRepository basketRepository;

    @Captor
    private ArgumentCaptor<Basket> basketArgumentCaptor;

    @Test
    void AddArticleToBasket() {
        // Arrange
        Article article;
        BasketAddItemRequest request;

        article = Article.builder().name(name).price(price).ean(ean).build();
        request = BasketAddItemRequest.builder().basketId(0L).articleId(1L).amount(100).build();

        when(articleRepository.findById(request.getArticleId())).thenReturn(Optional.of(article));
        when(basketRepository.findById(request.getBasketId())).thenReturn(Optional.empty());

        //Act
        Basket i = shopService.AddArticleToBsket(request);

        verify(basketRepository, times(1)).findById(request.getBasketId());
        verify(articleRepository, times(1)).findById(request.getArticleId());
        verify(basketRepository, times(1)).save(i);

        verify(basketRepository).save(basketArgumentCaptor.capture());
        Basket basket = basketArgumentCaptor.getValue();

        //Assert
        Assertions.assertEquals(1, basket.getItems().size());
        Assertions.assertEquals(article, i.getItems().get(0).getArticle());
    }

    @Test
    void RemoveArticleFromBasket(){
        // Arrange
        Article article = Article.builder().name(name).price(price).ean(ean).id(1L).build();
        CardItem cardItem = CardItem.builder().article(article).amount(2).id(1L).build();
        Basket basket = Basket.builder().payed(false).id(1L).build();
        basket.addItem(cardItem);

        when(basketRepository.findById(basket.getId())).thenReturn(Optional.of(basket));
        when(cardRepository.findById(cardItem.getId())).thenReturn(Optional.of(cardItem));

        //Act
        Basket resultBasket = shopService.RemoveArticleFromBasket(basket.getId(), cardItem.getId());

        //Assert
        verify(basketRepository, times(1)).findById(basket.getId());
        verify(cardRepository, times(1)).findById(cardItem.getId());
        verify(basketRepository, times(1)).save(basket);
        Assertions.assertEquals(basket, resultBasket);
    }

    @Test
    void GetBasketFound(){
        // Arrange
        Article article = Article.builder().name(name).price(price).ean(ean).id(1L).build();
        CardItem cardItem = CardItem.builder().article(article).amount(2).id(1L).build();
        Basket basket = Basket.builder().payed(true).id(1L).build();
        basket.addItem(cardItem);

        when(basketRepository.findById(1L)).thenReturn(Optional.of(basket));

        //Act
        Basket resultBasket = shopService.GetBasket(basket.getId());

        //Assert
        verify(basketRepository, times(1)).findById(basket.getId());
        Assertions.assertEquals(basket, resultBasket);
    }

    @Test
    void GetBasketNotFound(){
        // Arrange
        when(basketRepository.findById(2L)).thenReturn(Optional.empty());

        //Act
        Basket resultBasket = shopService.GetBasket(2L);
        verify(basketRepository, times(1)).findById(2L);

        //Assert
        Assertions.assertEquals(null, resultBasket);
    }

    @Test
    void PayingBasketReturnTrue(){
        // Arrange
        Article article = Article.builder().name(name).price(price).ean(ean).id(1L).build();
        CardItem cardItem = CardItem.builder().article(article).amount(2).id(1L).build();
        Basket basket = Basket.builder().payed(false).id(1L).build();
        basket.addItem(cardItem);

        when(basketRepository.findById(1L)).thenReturn(Optional.of(basket));

        //Act
        boolean result = shopService.PayingBasket(basket.getId());

        //Assert
        verify(basketRepository, times(1)).save(basket);
        Assertions.assertEquals(true, basket.isPayed());
        Assertions.assertEquals(true, result);
    }


    @Test
    void PayingBasketReturnFalse(){
        // Arrange
        Article article = Article.builder().name(name).price(price).ean(ean).id(1L).build();
        CardItem cardItem = CardItem.builder().article(article).amount(2).id(1L).build();
        Basket basket = Basket.builder().payed(true).id(1L).build();
        basket.addItem(cardItem);

        when(basketRepository.findById(1L)).thenReturn(Optional.of(basket));

        //Act
        boolean result = shopService.PayingBasket(basket.getId());

        //Assert
        verify(basketRepository, times(0)).save(basket);
        Assertions.assertEquals(true, basket.isPayed());
        Assertions.assertEquals(false, result);
    }
}
